function wavelet=get_wavelet(data,t1,t2,g1,g2,shift_max)
%  Copyright (C) 2010 Center for Subsurface Imaging and Fluid Modeling (CSIM),
%  King Abdullah University of Science and Technology, All rights reserved.
%
%  author:   Xin Wang
%  email:    xin.wang@kaust.edu.sa
%  date:     Sep 26, 2012
%
%  WAVELET: get the wavelet from data 
%
%  wavelet=get_wavelet(data)
%
%  IN   data(:,:) : data
%       t1        : up limit of time
%       t2        : down limit of time
%       g1        : left limit of trace
%       g2        : right limit of trace
%       shift_max : the maximum shift position could be
%
%  OUT  data(:,:) : Output wavelet 
%
%  Example : 
%
%    data=zeros(61,21); dt=0.004;freq=15;x=[1:21];t=[0:60]*dt;
%    wavelet0=ricker(freq,dt);nt=length(wavelet0);
%    for ig=1:11, data(11:10+nt,ig)=wavelet0; end
%    for ig=[12:13,15:21], data(14:13+nt,ig)=wavelet0; end
%    data(21:20+nt,14)=wavelet0;
%    wavelet=get_wavelet(data,11,10+nt,5,15,5);
%    figure;subplot(121);wiggle(x,t,data,'wiggle',0.5);title('input data'); 
%    subplot(122); plot(wavelet0,[0:nt-1]*dt,'-ok');hold on; 
%    plot(wavelet+1,[0:nt-1]*dt,'-*b');set(gca,'xtick',[],'ydir','reverse');
%    legend('true wavelet','calculated wavelet','Location','south');legend boxoff;

if nargin==5
    shift_max=0; % not allow trace to shift
end

data=data(t1:t2,g1:g2);
[nt,ng]=size(data);

if mod(ng,2)==0
    ig0=ng/2;
else
    ig0=(ng+1)/2;
end
amp_max=max(abs(squeeze(data(:,ig0))));

for ig =1:ng
    data(:,ig)=normalize(data(:,ig));
end
trace0=data(:,ig0);

trace_add=trace0;
ntrace_add=1;
for ig=1:ng
    if ig~=ig0
        trace1=data(:,ig);
        trace1_shift=zeros(size(trace0));
        trace_xcorr=xcorr(trace0,trace1);
        imax=max_loc(trace_xcorr);
        n_shift=imax-nt;
        if abs(n_shift)<shift_max && n_shift~=0
            n1=max(1,1+(n_shift));
            n2=min(nt,nt+(n_shift));
            n=n2-n1+1; 
            if n_shift>0
               trace1_shift(n1:n2)=trace1(1:n);
            else
               trace1_shift(1:n)=trace1(nt-n+1:end);
            end
            trace_add=trace_add+trace1_shift;
            ntrace_add=ntrace_add+1;
        elseif n_shift==0
            trace1_shift=trace1;         
            trace_add=trace_add+trace1_shift;
            ntrace_add=ntrace_add+1;
        end
    end
end

wavelet=trace_add/ntrace_add*amp_max;

end