function ttt=ttt_const2d(v,dx,dz,sx,sz)
%  Copyright (C) 2010 Center for Subsurface Imaging and Fluid Modeling (CSIM),
%  King Abdullah University of Science and Technology, All rights reserved.
%
%  author:   Xin Wang
%  email:    xin.wang@kaust.edu.sa
%  date:     Sep 26, 2012
%
%  ttt_const: get the traveltime table for constant velocity model 
%
%  ttt=ttt_const2d(v,dx,dz,sx,sz);
%
%  IN   v(:,:) : velocity model
%       dx     : grid interval in X direction
%       dz     : grid interval in Z direction
%       sx     : source X coordinates vector
%       sz     : source Z coordinates vector
%  OUT  ttt(:,:) : Output traveltime table for constant velocity with
%                  sources at [sz;sx]; 
%
%  Example :
%
%    nx = 401; nz = 201; dx = 5.0; dz = dx; x = (0:nx-1)*dx; z = (0:nz-1)*dx;
%    v = ones(nz,nx)*1500.0; sx=(0:2:(nx-1))*dx; sz=zeros(size(sx));
%    ttt=ttt_const2d(v,dx,dz,sx,sz);
%    figure;subplot(221);imagesc(x,z,v); 
%    title(['velocity model with constant v=',num2str(min(v(:))),'m/s']);
%    subplot(222); imagesc(x,z,squeeze(ttt(:,:,1))); hold on; plot(sx(1),sz(1),'w*');
%    title(['source point at (',num2str(sx(1)),',',num2str(sz(1)),')']);hold off;
%    subplot(223); imagesc(x,z,squeeze(ttt(:,:,51))); hold on; plot(sx(51),sz(51),'w*');
%    title(['source point at (',num2str(sx(51)),',',num2str(sz(51)),')']);hold off;
%    subplot(224); imagesc(x,z,squeeze(ttt(:,:,101))); hold on; plot(sx(101),sz(101),'w*');
%    title(['source point at (',num2str(sx(101)),',',num2str(sz(101)),')']);hold off;

[nz,nx]=size(v);
vmin=min(v(:));
vmax=max(v(:));
if vmin~=vmax
    error('Velocity is not constant!');
else
    vconst=vmin;
end

X=(0:nx-1)*dx;Z=(0:nz-1)*dz;
[XX,ZZ]=meshgrid(X,Z);

ns=length(sx);

ttt=zeros(nz,nx,ns);
for is=1:ns
    dist=((sz(is)-ZZ).^2+(sx(is)-XX).^2).^(0.5);
    ttt(:,:,is)=dist/vconst;
end

end
