!! module do 2-8 finite-difference computations for attenuation
module a2d_atten

 use global
 use datatype
 use math
 use string
 use io

 implicit none

contains

!----------------------------------------------------------------------------------------------
!! Solving the first-order constant-density visco-acoustic wave equation using a staggered grid
!! finite-difference method.

subroutine atten_modeling(is,par,coord,s,c,den,fs,nx_pml,nz_pml,npml,qf,wave,seis)

 type(param),       intent(in)  :: par
 type(acquisition), intent(in)  :: coord
 integer,           intent(in)  :: is,fs(:),nx_pml,nz_pml,npml
 real,              intent(in)  :: c(:,:),den(:,:),qf(:,:),s(:)
 real,              intent(out) :: seis(:,:),wave(:,:,:)

 real,              allocatable :: u(:,:),w(:,:),p(:,:)
 real,              allocatable :: e1(:),e2(:),e51(:),e52(:),e91(:),e92(:)
 real,              allocatable :: rl(:,:),rl1(:,:),tausigma(:,:),tauepsilon(:,:)
 integer                        :: isx,isz,igx,igz,iz,ix,it,ig
 real                           :: alpha,beta,kappa

 !! Initialize the outputs
 !! ----------------------
 seis=0.0
 !$omp parallel do private(iz,ix)
 do ix=1,par%nx
 do iz=1,par%nz
    wave(iz,ix,1:par%nt)=0.0
 enddo
 enddo
 !$omp end parallel do

 !! Memory allocations
 !! ------------------
 allocate(e1(par%npml),e2(par%npml),e51(par%npml),e52(par%npml),e91(par%npml),e92(par%npml))
 allocate(u(nz_pml,nx_pml),w(nz_pml,nx_pml),p(nz_pml,nx_pml),rl(nz_pml,nx_pml),rl1(nz_pml,nx_pml))
 allocate(tausigma(nz_pml,nx_pml),tauepsilon(nz_pml,nx_pml))
 e1=0.0;  e2=0.0;  e51=0.0;  e52=0.0;  e91=0.0;  e92=0.0
 !$omp parallel do private(iz,ix)
 do ix=1,nx_pml
 do iz=1,nz_pml
    u(iz,ix)=0.0
    w(iz,ix)=0.0
    p(iz,ix)=0.0
    rl(iz,ix)=0.0
    rl1(iz,ix)=0.0
    tausigma(iz,ix)=0.0
    tauepsilon(iz,ix)=0.0
 enddo
 enddo
 !$omp end parallel do

 !! Set up absorbing boundaries
 !! ---------------------------
 call pmlcoef(par%npml,e1,e2,e51,e52,e91,e92)

 !! Set up stress and strain relaxation parameters
 !! ----------------------------------------------
 !$OMP PARALLEL DO private(iz,ix)
 do ix=1,nx_pml
 do iz=1,nz_pml
    tausigma(iz,ix)=(sqrt(1.0+1.0/qf(iz,ix)/qf(iz,ix))-1.0/qf(iz,ix))/(2.0*pi*par%f)
    tauepsilon(iz,ix)=1.0/(2.0*pi*par%f)/(2.0*pi*par%f)/tausigma(iz,ix)
 enddo
 enddo
 !$OMP END PARALLEL DO

 !! Set up source position
 !! ----------------------
 isx=npml+int(coord%xs(is)/par%dx)+1
 isz=npml+int(coord%zs(is)/par%dx)+1
 if (isz==fs(isx)) isz=isz + 1

 !! Time marching loop
 !! ------------------
 do it=1,par%nt

  !! Update the pressure field (interior)
  !! ------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    p(iz,ix)=p(iz,ix)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*                   &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (left)
  !! --------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    p(iz,ix)=p(iz,ix)*e1(ix-2)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*          &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(ix-2)
  enddo
  enddo
  !$OMP END PARALLEL DO 

  !! Update the pressure field (top)
  !! -------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=5,npml
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    p(iz,ix)=p(iz,ix)*e1(iz-2)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*          &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(iz-2)
  enddo
  enddo
  !OMP END PARALLEL DO 

  !! Update the pressure field (bottom)
  !! -------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=nz_pml-npml+1,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    p(iz,ix)=p(iz,ix)*e1(nz_pml-iz-1)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*   &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (right)
  !! ---------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    p(iz,ix)=p(iz,ix)*e1(nx_pml-ix-1)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*     &
               (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
                c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
                c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
                c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(nx_pml-ix-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the memory variable
  !! --------------------------
  !$omp parallel do private(iz,ix)
  do ix=1,nx_pml
  do iz=1,nz_pml
    rl1(iz,ix)=rl(iz,ix)
  enddo
  enddo
  !$omp end parallel do

  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,nx_pml-4
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*(tauepsilon(iz,ix)/tausigma(iz,ix)-1.0)/(par%dx)/tausigma(iz,ix)
    rl(iz,ix)=((1.0-par%dt/2.0/tausigma(iz,ix))*rl1(iz,ix)-alpha*                 &
              (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
               c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
               c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
               c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix))))/(1.0+par%dt/2.0/tausigma(iz,ix))
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Add source
  !! ----------
  beta=den(isz,isx)*c(isz,isx)*c(isz,isx)*par%dt
  p(isz,isx)=p(isz,isx)+beta*s(it)

  !! Update particle velocity: u, w (interior)
  !! -----------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +    &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+   &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+   &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

     w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
                                     
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (left)
  !! -------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
  do iz=5,npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=npml+1,nz_pml-npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                             c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                             c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                             c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix))+   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (right)
  !! --------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
  do iz=5,npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix))+   &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=npml+1,nz_pml-npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                             c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                             c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                             c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (top)
  !! ------------------------------------ 
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=5,npml
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=par%dt/(den(iz,ix)*par%dx)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Output pressure seismogram 
  !! --------------------------
  !$omp parallel do private(ig,igx,igz)
  do ig=1,coord%ng(is)
    igx=npml+int(coord%xg(is,ig)/par%dx)+1
    igz=npml+int(coord%zg(is,ig)/par%dx)+1
    if (igz.eq.fs(igx)) igz=igz+1
     seis(it,ig)=p(igz,igx)
  enddo
  !$omp end parallel do

  !if(mod(it,50).eq.0) then
  !      call filename(output,'snapshot_100q',it,'.bin')
  !      call write_binfile(output,p,nz_pml,nx_pml)
  !endif

  if(par%ic==0) then

    !$omp parallel do private(iz,ix)
    do ix=npml+1,nx_pml-npml
    do iz=npml+1,nz_pml-npml
        wave(iz-npml,ix-npml,it)=p(iz,ix)
    enddo 
    enddo
    !$omp end parallel do

  else

    !$omp parallel do private(iz,ix,alpha)
    do ix=npml+1,nx_pml-npml
    do iz=npml+1,nz_pml-npml
        alpha = tauepsilon(iz,ix)/tausigma(iz,ix)
        wave(iz-npml,ix-npml,it)=-alpha*(c1_staggered_8th*(u(iz,ix)-u(iz,ix-1)+w(iz,ix)-w(iz-1,ix)) +     &
                                         c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2)+w(iz+1,ix)-w(iz-2,ix)) + &
                                         c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3)+w(iz+2,ix)-w(iz-3,ix)) + &
                                         c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4)+w(iz+3,ix)-w(iz-4,ix)))
    enddo
    enddo
    !$omp end parallel do
  endif

 enddo  !! End of time-marching loop

 999 continue
 deallocate(e1,e2,e51,e52,e91,e92)
 deallocate(u,w,p,rl,rl1)
 deallocate(tausigma,tauepsilon)

end subroutine atten_modeling


!---------------------------------------------
subroutine pmlcoef(npml,e1,e2,e51,e52,e91,e92)

 implicit none
 integer,    intent(in)  :: npml
 integer                 :: ix
 real                    :: fac,myfac,x
 real,       intent(out) :: e1(:),e2(:),e51(:),e52(:),e91(:),e92(:)

 fac=0.88
 myfac=acos(fac)/real(npml)
 do ix=1,npml
    x=real(npml-ix)
    e1(ix)=cos(myfac*x)
    e2(ix)=sqrt(cos(myfac*x))
    x=real(npml-ix)+0.5
    e51(ix)=cos(myfac*x)
    e52(ix)=sqrt(cos(myfac*x))
 enddo

 do ix=1,npml-1
    x=real(npml-ix)-0.5
    e91(ix)=cos(myfac*x)
    e92(ix)=sqrt(cos(myfac*x))
 enddo

 e91(npml)=1.0
 e92(npml)=1.0

end subroutine pmlcoef


!------------------------------------------------
subroutine mute_direct(par,coord,is,v,taper,seis)

 implicit none
 type(acquisition), intent(in) :: coord
 type(param),       intent(in) :: par
 integer,           intent(in) :: is
 real,              intent(in) :: v(:,:),taper(:)
 real,           intent(inout) :: seis(:,:)
 real                          :: t_shift,distance
 integer                       :: ig,it,ishift,isx,isz

 isx = par%npml+int(coord%xs(is)/par%dx)+1
 isz = par%npml+int(coord%zs(is)/par%dx)+1

 do ig=1,coord%ng(is)
    distance = (coord%xs(is)-coord%xg(is,ig))**2+(coord%zs(is)-coord%zg(is,ig))**2
    t_shift = sqrt(distance)/v(isz,isx)
    
    ishift = int(t_shift/par%dt)+par%window_size
    seis(1:ishift,ig)=0.0
    do it=ishift+1,ishift+par%n_taper
       seis(it,ig)=seis(it,ig)*taper(it-ishift)
    enddo
 enddo
   
end subroutine mute_direct


!--------------------------
subroutine hanning(taper,n)

 implicit none
 real,intent(out)         :: taper(:)
 integer,intent(in)          :: n
 integer                     :: i
 real                        :: m

 m=2.0*n+1.0

 do i=1,n
    taper(i)=0.5*(1-cos(2*pi*i/m))
 enddo

 end subroutine hanning

end module a2d_atten

