!! module do 2-8 finite-difference computations for attenuation
module a2d_atten_wq

 use global
 use datatype
 use math
 use string
 use io
 use a2d_atten

 implicit none

contains

!----------------------------------------------------------------------------------------------
!! Solving the first-order constant-density visco-acoustic wave equation using a staggered grid
!! finite-difference method.

subroutine atten_modeling_wq(is,par,coord,s,c,den,fs,nx_pml,nz_pml,npml,qf, &
                             tau,tausigma,tauepsilon,save_wavefield,wave,seis)

 type(param),       intent(in)  :: par
 type(acquisition), intent(in)  :: coord
 integer,           intent(in)  :: is,fs(:),nx_pml,nz_pml,npml
 real,              intent(in)  :: c(:,:),den(:,:),qf(:,:),s(:), &
                                   tau(:,:),tausigma(:,:),tauepsilon(:,:)
 real,              intent(out) :: seis(:,:),wave(:,:,:)
 logical,           intent(in)  :: save_wavefield

 real,              allocatable :: u(:,:),w(:,:),p(:,:)
 real,              allocatable :: e1(:),e2(:),e51(:),e52(:),e91(:),e92(:)
 real,              allocatable :: rl(:,:),rl1(:,:)
 integer                        :: isx,isz,igx,igz,iz,ix,it,ig
 real                           :: alpha,beta,kappa,dtdx

 !! Memory allocations
 !! ------------------
 allocate(e1(par%npml),e2(par%npml),e51(par%npml),e52(par%npml),e91(par%npml),e92(par%npml))
 allocate(u(nz_pml,nx_pml),w(nz_pml,nx_pml),p(nz_pml,nx_pml),rl(nz_pml,nx_pml),rl1(nz_pml,nx_pml))
 e1=0.0;  e2=0.0;  e51=0.0;  e52=0.0;  e91=0.0; e92=0.0

 !$omp parallel do private(iz,ix)
 do ix=1,nx_pml
 do iz=1,nz_pml
    u(iz,ix)=0.0
    w(iz,ix)=0.0
    p(iz,ix)=0.0
    rl(iz,ix)=0.0
    rl1(iz,ix)=0.0
 enddo
 enddo
 !$omp end parallel do

 !! Initialize the outputs
 !! ----------------------
 seis=0.0
 if(save_wavefield) then
    !$omp parallel do private(iz,ix)
    do ix=1,par%nx
    do iz=1,par%nz
        wave(iz,ix,1:par%nt)=0.0
    enddo
    enddo
    !$omp end parallel do
 endif

 !! Set up absorbing boundaries
 !! ---------------------------
 call pmlcoef(par%npml,e1,e2,e51,e52,e91,e92)

 !! Set up source position
 !! ----------------------
 isx=npml+int(coord%xs(is)/par%dx)+1
 isz=npml+int(coord%zs(is)/par%dx)+1
 if (isz==fs(isx)) isz=isz+1

 dtdx=par%dt/par%dx !! Local variable for reducing flops

 !! Time marching loop
 !! ------------------
 do it=1,par%nt

  !! Update the pressure field (interior)
  !! ------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    !alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*tauepsilon(iz,ix)/tausigma(iz,ix)
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p(iz,ix)=p(iz,ix)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*                   &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (left)
  !! --------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p(iz,ix)=p(iz,ix)*e1(ix-2)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*          &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(ix-2)
  enddo
  enddo
  !$OMP END PARALLEL DO 

  !! Update the pressure field (top)
  !! -------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=5,npml
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p(iz,ix)=p(iz,ix)*e1(iz-2)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*          &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(iz-2)
  enddo
  enddo
  !OMP END PARALLEL DO 

  !! Update the pressure field (bottom)
  !! -------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=nz_pml-npml+1,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p(iz,ix)=p(iz,ix)*e1(nz_pml-iz-1)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*   &
             (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
              c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
              c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
              c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (right)
  !! ---------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p(iz,ix)=p(iz,ix)*e1(nx_pml-ix-1)-0.5*(rl(iz,ix)+rl1(iz,ix))*par%dt-alpha*     &
               (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
                c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
                c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
                c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix)))*e2(nx_pml-ix-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the memory variable
  !! --------------------------
  !$omp parallel do private(iz,ix)
  do ix=1,nx_pml
  do iz=1,nz_pml
    rl1(iz,ix)=rl(iz,ix)
  enddo
  enddo
  !$omp end parallel do

  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,nx_pml-4
  do iz=5,nz_pml-4
    !alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tauepsilon(iz,ix)/tausigma(iz,ix)-1.0)/tausigma(iz,ix)
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*tau(iz,ix)/tausigma(iz,ix)
    rl(iz,ix)=((1.0-par%dt/2.0/tausigma(iz,ix))*rl1(iz,ix)-alpha*                 &
              (c1_staggered_8th*(u(iz,ix)-u(iz,ix-1) + w(iz,ix)-w(iz-1,ix)) +     &
               c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2) + w(iz+1,ix)-w(iz-2,ix)) + &
               c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3) + w(iz+2,ix)-w(iz-3,ix)) + &
               c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4) + w(iz+3,ix)-w(iz-4,ix))))/(1.0+par%dt/2.0/tausigma(iz,ix))
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Add source
  !! ----------
  beta=den(isz,isx)*c(isz,isx)*c(isz,isx)*par%dt
  p(isz,isx)=p(isz,isx)+beta*s(it)

  !! Update particle velocity: u, w (interior)
  !! -----------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +    &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+   &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+   &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

     w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
                                     
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (left)
  !! -------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
  do iz=5,npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=npml+1,nz_pml-npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                             c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                             c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                             c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                       c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                       c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                       c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e52(ix-2)

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix))+   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (right)
  !! --------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
  do iz=5,npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix))+   &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix))+ &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix))+ &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=npml+1,nz_pml-npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                             c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                             c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                             c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                                              c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                                              c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                                              c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))*e92(nx_pml-ix-1)

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (top)
  !! ------------------------------------ 
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=5,npml
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

    w(iz,ix)=w(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                       c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                       c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                       c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e52(iz-2)
  enddo

  do iz=nz_pml-npml+1,nz_pml-4
    alpha=dtdx/den(iz,ix)
    u(iz,ix)=u(iz,ix)-alpha*(c1_staggered_8th*(p(iz,ix+1)-p(iz,ix)) +  &
                             c2_staggered_8th*(p(iz,ix+2)-p(iz,ix-1))+ &
                             c3_staggered_8th*(p(iz,ix+3)-p(iz,ix-2))+ &
                             c4_staggered_8th*(p(iz,ix+4)-p(iz,ix-3)))

    w(iz,ix)=w(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p(iz+1,ix)-p(iz,ix)) +   &
                                              c2_staggered_8th*(p(iz+2,ix)-p(iz-1,ix)) + &
                                              c3_staggered_8th*(p(iz+3,ix)-p(iz-2,ix)) + &
                                              c4_staggered_8th*(p(iz+4,ix)-p(iz-3,ix)))*e92(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Output pressure seismogram 
  !! --------------------------
  !$omp parallel do private(ig,igx,igz)
  do ig=1,coord%ng(is)
    igx=npml+int(coord%xg(is,ig)/par%dx)+1
    igz=npml+int(coord%zg(is,ig)/par%dx)+1
    if (igz.eq.fs(igx)) igz=igz+1
     seis(it,ig)=p(igz,igx)
  enddo
  !$omp end parallel do

  if(save_wavefield) then
        !$omp parallel do private(iz,ix,alpha)
        do ix=npml+1,nx_pml-npml
        do iz=npml+1,nz_pml-npml
            !alpha = den(iz,ix)*c(iz,ix)*c(iz,ix)*(1.0+1.0/tausigma(iz,ix))	!! Changed this temporarily
            alpha=1.0
            wave(iz-npml,ix-npml,it)=-alpha*(c1_staggered_8th*(u(iz,ix)-u(iz,ix-1)+w(iz,ix)-w(iz-1,ix)) +     &
                                         c2_staggered_8th*(u(iz,ix+1)-u(iz,ix-2)+w(iz+1,ix)-w(iz-2,ix)) + &
                                         c3_staggered_8th*(u(iz,ix+2)-u(iz,ix-3)+w(iz+2,ix)-w(iz-3,ix)) + &
                                         c4_staggered_8th*(u(iz,ix+3)-u(iz,ix-4)+w(iz+3,ix)-w(iz-4,ix)))
        enddo
        enddo
        !$omp end parallel do
  endif

 enddo  !! End of time-marching loop

 999 continue
 deallocate(e1,e2,e51,e52,e91,e92)
 deallocate(u,w,p,rl,rl1)

end subroutine atten_modeling_wq

!!-------------------------------------------------------------------------------------------------------------
!! Solving the first-order constant-density visco-acoustic wave equation in reverse time using a staggered grid
!! finite-difference method.
subroutine atten_rtm_wq(is,par,coord,s,c,den,fs,nx_pml,nz_pml,npml,wave,qf, &
                        tau,tausigma,tauepsilon,seis,energy,mig)

 type(param),       intent(in)  :: par
 type(acquisition), intent(in)  :: coord
 integer,           intent(in)  :: is,fs(:),nx_pml,nz_pml,npml
 real,              intent(in)  :: c(:,:),den(:,:),s(:)
 real,              intent(in)  :: seis(:,:),wave(:,:,:),qf(:,:), &
                                   tau(:,:),tausigma(:,:),tauepsilon(:,:)
 real,             intent(inout):: mig(:,:),energy(:,:)

 real,              allocatable :: u1(:,:),w1(:,:),p1(:,:),rl1(:,:),rl11(:,:)
 real,              allocatable :: e1(:),e2(:),e51(:),e52(:),e91(:),e92(:)
 integer                        :: isx,isz,igx,igz,iz,ix,it,ig
 real                           :: alpha,beta,kappa,dtdx

 !! Memory allocations
 !! ------------------ 
 allocate(u1(nz_pml,nx_pml),w1(nz_pml,nx_pml),p1(nz_pml,nx_pml),rl1(nz_pml,nx_pml),rl11(nz_pml,nx_pml))
 allocate(e1(par%npml),e2(par%npml),e51(par%npml),e52(par%npml),e91(par%npml),e92(par%npml))
 e1=0.0;  e2=0.0;  e51=0.0;  e52=0.0;  e91=0.0; e92=0.0

 !$omp parallel do private(iz,ix)
 do ix=1,nx_pml
 do iz=1,nz_pml
    u1(iz,ix)=0.0
    w1(iz,ix)=0.0
    p1(iz,ix)=0.0
    rl1(iz,ix)=0.0
    rl11(iz,ix)=0.0
 enddo
 enddo
 !$omp end parallel do 

 !! Initialize the outputs
 !! ----------------------
 !$omp parallel do private(iz,ix)
 do ix=1,nx_pml
 do iz=1,nz_pml
    mig(iz,ix)=0.0
    energy(iz,ix)=0.0
 enddo
 enddo
 !$omp end parallel do

 !! Set up absorbing boundaries
 !! ---------------------------
 call pmlcoef(par%npml,e1,e2,e51,e52,e91,e92)

 dtdx=par%dt/par%dx !! Local variable for reducing flops

 !! Time marching loop 
 !! ------------------
 do it=par%nt,1,-1

  !! Update the pressure field (interior)
  !! ------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    !alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*tauepsilon(iz,ix)/(par%dx)/tausigma(iz,ix)
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p1(iz,ix)=p1(iz,ix)-0.5*(rl1(iz,ix)+rl11(iz,ix))*par%dt-alpha*                        &
                    (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +     &
                     c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                     c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                     c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix)))
             
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (left)
  !! --------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p1(iz,ix)=p1(iz,ix)*e1(ix-2)-0.5*(rl1(iz,ix)+rl11(iz,ix))*par%dt-alpha*           &
                (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +     &
                 c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                 c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                 c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix)))*e2(ix-2)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (top)
  !! -------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=5,npml
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p1(iz,ix)=p1(iz,ix)*e1(iz-2)-0.5*(rl1(iz,ix)+rl11(iz,ix))*par%dt-alpha*           &
                (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +     &
                 c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                 c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                 c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix)))*e2(iz-2)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (bottom)
  !! ----------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=nz_pml-npml+1,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p1(iz,ix)=p1(iz,ix)*e1(nz_pml-iz-1)-0.5*(rl1(iz,ix)+rl11(iz,ix))*par%dt-alpha*    &
                 (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +    &
                 c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                 c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                 c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix)))*e2(nz_pml-iz-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the pressure field (right)
  !! ---------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
  do iz=5,nz_pml-4
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*(tau(iz,ix)+1.0)
    p1(iz,ix)=p1(iz,ix)*e1(nx_pml-ix-1)-0.5*(rl1(iz,ix)+rl11(iz,ix))*par%dt-alpha*    &
                (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +     &
                 c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                 c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                 c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix)))*e2(nx_pml-ix-1)
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update the memory variable
  !! --------------------------
  !$omp parallel do private(iz,ix)
  do ix=1,nx_pml
  do iz=1,nz_pml
    rl11(iz,ix)=rl1(iz,ix)
  enddo
  enddo
  !$omp end parallel do

  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,nx_pml-4
  do iz=5,nz_pml-4
    !alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*par%dt*(tauepsilon(iz,ix)/tausigma(iz,ix)-1.0)/(par%dx)/tausigma(iz,ix)
    alpha=den(iz,ix)*c(iz,ix)*c(iz,ix)*dtdx*tau(iz,ix)/tausigma(iz,ix)
    rl1(iz,ix)=((1.0-par%dt/2.0/tausigma(iz,ix))*rl11(iz,ix)-alpha*                   &
                (c1_staggered_8th*(u1(iz,ix)-u1(iz,ix-1)+w1(iz,ix)-w1(iz-1,ix)) +     &
                 c2_staggered_8th*(u1(iz,ix+1)-u1(iz,ix-2)+w1(iz+1,ix)-w1(iz-2,ix)) + &
                 c3_staggered_8th*(u1(iz,ix+2)-u1(iz,ix-3)+w1(iz+2,ix)-w1(iz-3,ix)) + &
                 c4_staggered_8th*(u1(iz,ix+3)-u1(iz,ix-4)+w1(iz+3,ix)-w1(iz-4,ix))))/(1.0+par%dt/2.0/tausigma(iz,ix))
  enddo
  enddo
  !$OMP END PARALLEL DO 

  !! Add seismogram
  !! ---------------
  !$omp parallel do private(ig,igx,igz,beta)
  do ig=1,coord%ng(is)
     igx=npml+int(coord%xg(is,ig)/par%dx)+1
     igz=npml+int(coord%zg(is,ig)/par%dx)+1
     !! Geophone position is shifted downward 1 node if it is on the free surface
     !! -------------------------------------------------------------------------
     if (igz==fs(igx)) igz=igz+1
     beta=den(igz,igx)*c(igz,igx)*c(igz,igx)*par%dt
     p1(igz,igx)=p1(igz,igx)+beta*seis(it,ig)
  enddo
  !$omp end parallel do

  !! Update particle velocity: u, w (interior)
  !! -----------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    alpha=dtdx/den(iz,ix)
    u1(iz,ix)=u1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                               c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                               c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                               c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))

    w1(iz,ix)=w1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                               c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                               c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                               c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))
  enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (left)
  !! -------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=5,npml
    do iz=5,npml
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                             c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                             c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                             c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e52(ix-2)

        w1(iz,ix)=w1(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                             c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                             c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                             c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e52(iz-2)
    enddo

    do iz=npml+1,nz_pml-npml
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                             c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                             c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                             c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e52(ix-2)

        w1(iz,ix)=w1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                   c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                   c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                   c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))
    enddo

    do iz=nz_pml-npml+1,nz_pml-4
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e51(ix-2)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                             c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                             c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                             c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e52(ix-2)

        w1(iz,ix)=w1(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                                    c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                                    c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                                    c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e92(nz_pml-iz-1)
    enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (right)
  !! --------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=nx_pml-npml+1,nx_pml-4
    do iz=5,npml
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                                    c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                                    c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                                    c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e92(nx_pml-ix-1)

        w1(iz,ix)=w1(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                             c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                             c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                             c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e52(iz-2)
    enddo

    do iz=npml+1,nz_pml-npml
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                                    c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                                    c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                                    c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e92(nx_pml-ix-1)
      
        w1(iz,ix)=w1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                   c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                   c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                   c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))
    enddo

    do iz=nz_pml-npml+1,nz_pml-4
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)*e91(nx_pml-ix-1)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                                    c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                                    c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                                    c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))*e92(nx_pml-ix-1)

        w1(iz,ix)=w1(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                                    c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                                    c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                                    c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e92(nz_pml-iz-1)
    enddo
  enddo
  !$OMP END PARALLEL DO

  !! Update particle velocity: u, w (top)
  !! ------------------------------------
  !$OMP PARALLEL DO private(iz,ix,alpha)
  do ix=npml+1,nx_pml-npml
    do iz=5,npml
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                   c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                   c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                   c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))

        w1(iz,ix)=w1(iz,ix)*e51(iz-2)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                             c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                             c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                             c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e52(iz-2)
    enddo

    do iz=nz_pml-npml+1,nz_pml-4
        alpha=dtdx/den(iz,ix)
        u1(iz,ix)=u1(iz,ix)-alpha*(c1_staggered_8th*(p1(iz,ix+1)-p1(iz,ix)) +  &
                                   c2_staggered_8th*(p1(iz,ix+2)-p1(iz,ix-1))+ &
                                   c3_staggered_8th*(p1(iz,ix+3)-p1(iz,ix-2))+ &
                                   c4_staggered_8th*(p1(iz,ix+4)-p1(iz,ix-3)))

        w1(iz,ix)=w1(iz,ix)*e91(nz_pml-iz-1)-alpha*(c1_staggered_8th*(p1(iz+1,ix)-p1(iz,ix)) +   &
                                                    c2_staggered_8th*(p1(iz+2,ix)-p1(iz-1,ix)) + &
                                                    c3_staggered_8th*(p1(iz+3,ix)-p1(iz-2,ix)) + &
                                                    c4_staggered_8th*(p1(iz+4,ix)-p1(iz-3,ix)))*e92(nz_pml-iz-1)
    enddo
  enddo
  !$OMP END PARALLEL DO

  !! Dot product imaging condition
  !! -----------------------------
  !$OMP PARALLEL DO private(iz,ix)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    mig(iz,ix)=mig(iz,ix)+p1(iz,ix)*wave(iz-npml,ix-npml,it)
    !mig(iz,ix)=mig(iz,ix)+p1(iz,ix)*wave(iz-npml,ix-npml,it) &
    !                     +wave(iz-npml,ix-npml,it)*0.5*(rl1(iz,ix)+rl11(iz,ix))/tausigma(iz,ix)
  enddo
  enddo
  !$OMP END PARALLEL DO
 
  !! Save illumination
  !! -----------------
  !$OMP PARALLEL DO private(iz,ix)
  do ix=npml+1,nx_pml-npml
  do iz=npml+1,nz_pml-npml
    energy(iz,ix)=energy(iz,ix)+wave(iz-npml,ix-npml,it)*wave(iz-npml,ix-npml,it)
  enddo
  enddo
  !$OMP END PARALLEL DO

 enddo  ! End of time-marching loop

 999 continue
 deallocate(u1,w1,p1,rl1,rl11)
 deallocate(e1,e2,e51,e52,e91,e92)

end subroutine atten_rtm_wq
!--------------------------

end module a2d_atten_wq

