! module containing I/O utilities
!
module io

use string
implicit none

interface read_sufile
  module procedure read_sufile1
  module procedure read_sufile_2d
end interface

interface read_binfile
  module procedure read_binfile_1d
  module procedure read_binfile_2d
  module procedure read_binfile_3d
end interface

interface read_binfile_intel
  module procedure read_binfile_intel_1d
  module procedure read_binfile_intel_2d
  module procedure read_binfile_intel_3d
end interface

interface write_sufile
  module procedure write_sufile_1d
  module procedure write_sufile_2d
  module procedure write_sufile_2d_coord
end interface

interface write_binfile
  module procedure write_binfile_1d
  module procedure write_binfile_2d
  module procedure write_binfile_3d
end interface

interface write_binfile_intel
  module procedure write_binfile_intel_1d
  module procedure write_binfile_intel_2d
  module procedure write_binfile_intel_3d
end interface

contains

!---------------------------------------------------------------------------------------------
! Rules for verifying the input parameters
!
! STRING:
!         - required parameter will have a default as 'n/a'
!         - optional parameter will have some default which is not 'n/a'
!
! INTEGER:
!         - required parameter will have a default as -1
!         - optional parameter will have some default which is not -1
!
! FLOATING-POINT:
!         - required parameter will have a default as -1.0
!         - optional parameter will have some default which is not -1.0
!
subroutine readparamfile(parfile, par)

use datatype
use parser
use mmi_mpi

character(len=*), intent(in) :: parfile
type(param), intent(out)     :: par
character(len=100)           :: tmp, velfile, coordfile
integer                      :: nx, nz, nt
real                         :: dx, dt, f, dt_out

if (rank.eq.0) then

  !! Parameters required for most simulations
  !! ----------------------------------------
  call readParFile(parfile, 'VEL_IN',           par%velfile,            'n/a')
  call readParFile(parfile, 'COORD_FILE',       par%coordfile,          'n/a')
  call readParFile(parfile, 'CSG_IN',           par%csg_in,             'n/a')
  call readParFile(parfile, 'ILLUM_FILE',       par%illumfile,          'n/a')
  call readParFile(parfile, 'Q_IN',             par%qualfile,           'n/a')
  call readParFile(parfile, 'SNAPSHOT_FILE',    par%snapshot_file,      'n/a')
  call readParFile(parfile, 'NX',               par%nx,                    -1)
  call readParFile(parfile, 'NZ',               par%nz,                    -1)
  call readParFile(parfile, 'NY',               par%ny,                    -1)
  call readParFile(parfile, 'NT_WORK',          par%nt,                    -1)
  call readParFile(parfile, 'DX',               par%dx,                  -1.0)
  call readParFile(parfile, 'DT_WORK',          par%dt,                  -1.0)
  call readParFile(parfile, 'FILEFORMAT',       par%fileformat,           'H')
  call readParFile(parfile, 'NPML',             par%npml,                  40)
  call readParFile(parfile, 'FREQUENCY',        par%f,                    5.0)
  call readParFile(parfile, 'FREQUENCY_MAX',    par%fmax,                25.0)
  call readParFile(parfile, 'STRATEGY',         par%strategy,      'snapshot')
  call readParFile(parfile, 'SKIPSHOT',         par%skipshot,               0)
  call readParFile(parfile, 'SKIPTRACE',        par%skiptrace,              0)
  call readParFile(parfile, 'FREESURFACE',      par%free_surface,           1)
  call readParFile(parfile, 'SOURCEFILE',       par%sourcefile,         'n/a')
  call readParFile(parfile, 'SOURCETYPE',       par%sourcetype,      'normal')

  !! LSRTM parameters
  !! ----------------
  call readParFile(parfile, 'HIGHPASS',         par%highpass,               0)
  call readParFile(parfile, 'MIG_FILE',         par%migfile,      'mig_final')
  call readParFile(parfile, 'MIG_FILE_SHOT',    par%migfile_shot,  'mig_shot')
  call readParFile(parfile, 'REFL_FILE',        par%reflfile,           'n/a')
  call readParFile(parfile, 'IMAGE_CONDITION',  par%ic,                     2)

  !! Parameters common for both LSRTM and FWI
  !! ----------------------------------------
  call readParFile(parfile, 'ITERMAX',          par%itermax,              100)
  call readParFile(parfile, 'MUTE_DIRECT',      par%mute_direct,            0)
  call readParFile(parfile, 'PRECONDITION',     par%pre,                    1)
  call readParFile(parfile, 'GRAD_FILE',        par%gradfile,           'n/a')
  call readParFile(parfile, 'LOGFILE',          par%logfile,       'logs/log')
  call readParFile(parfile, 'N_TAPER',          par%n_taper,               50)
  call readParFile(parfile, 'CSG_OUT',          par%csg_out,            'csg')
  call readParFile(parfile, 'ISMARINE',         par%ismarine,               0)
  call readParFile(parfile, 'WATERBOTTOMFILE',  par%waterbottomfile,    'n/a')
  call readParFile(parfile, 'RESIDUALFILE',     par%residualfile,   'csg_res')
  call readParFile(parfile, 'IZWB',             par%izwb,                  -1)
  call readParFile(parfile, 'WINDOW_SIZE',      par%window_size,          150)

  !! FWI parameters
  !! --------------
  call readParFile(parfile, 'NSEARCH_MAX',      par%nsearch_max,           10)
  call readParFile(parfile, 'LINE_SEARCH',      par%line_search,            1)
  call readParFile(parfile, 'NORMALIZE',        par%normalize,              0)
  call readParFile(parfile, 'SMOOTHGRAD',       par%smoothgrad,             0)
  call readParFile(parfile, 'SMOOTHVEL',        par%smoothvel,              0)
  call readParFile(parfile, 'VCONSTRAINT',      par%vconstraint,         -1.0)
  call readParFile(parfile, 'VEL_OUT',          par%velfile_out,        'vel')
  call readParFile(parfile, 'VMIN',             par%vmin,              1500.0)
  call readParFile(parfile, 'VMAX',             par%vmax,              4500.0)
  call readParFile(parfile, 'STEP',             par%step,                 5.0)
  call readParFile(parfile, 'SMOOTHING_WINDOW', par%smooth,                 5)
  call readParFile(parfile, 'CSG_OUT_RES',      par%csg_out_res,        'csg')
  call readParFile(parfile, 'PERC_GRAD',        par%perc_grad,          100.0)

  !! WQ parameters
  !! -------------
  call readParFile(parfile, 'TAU_MIN',          par%tau_min,           0.0002)
  call readParFile(parfile, 'TAU_MAX',          par%tau_max,             0.07)
  call readParFile(parfile, 'SMOOTH_TAU',       par%smooth_tau,             0)
  call readParFile(parfile, 'CSG_MOD',          par%csg_mod,        'csg_mod')
  call readParFile(parfile, 'INPUT_TAU',        par%input_tau,              0)
  call readParFile(parfile, 'TAUFILE',          par%taufile,            'n/a')
  call readParFile(parfile, 'TAU_CONSTRAINT',   par%tau_constraint,       0.0)
  

  !! Multisource parameters
  !! ----------------------
  call readParFile(parfile, 'ENCODING_CODES',   par%encoding_codes,     'n/a')
  call readParFile(parfile, 'ENCODING_POLARITY',par%encoding_polarity,  'n/a')

  !! Optional parameters (not used so far)
  !! -------------------------------------
  call readParFile(parfile, 'VEL_BG',           par%velfile_bg,   par%velfile)
  call readParFile(parfile, 'TOPO',             par%topofile,           'n/a')
  call readParFile(parfile, 'DATA',             par%data,          'pressure')
  call readParFile(parfile, 'DENSITYFILE',      par%densityfile,        'n/a')
  call readParFile(parfile, 'VPFILE',           par%vpfile,             'n/a')
  call readParFile(parfile, 'VSFILE',           par%vsfile,             'n/a')

  ! Required Integer parameters
  call readParFile(parfile, 'NT_IN',            par%nt_in,             par%nt)
  call readParFile(parfile, 'NT_OUT',           par%nt_out,            par%nt)
  call readParFile(parfile, 'SHIFT',            par%shift_wavefield,        0)
  call readParFile(parfile, 'VARIABLE_DENSITY', par%variable_density,       0)
  call readParFile(parfile, 'MUTE_DATA',        par%mute_data,              0)
  call readParFile(parfile, 'METHOD',           par%method,                 1)
  call readParFile(parfile, 'EXPORTWAVEFIELD',  par%exportwavefield,       -1)

  ! Floating-point parameters
  call readParFile(parfile, 'DT_IN',            par%dt_in,             par%dt)
  call readParFile(parfile, 'DT_OUT',           par%dt_out,            par%dt)
  call readParFile(parfile, 'XMIN',             par%xmin,                 0.0)
  call readParFile(parfile, 'OFFSET_MIN',       par%offset_min,           0.0)
  call readParFile(parfile, 'WINDOW',           par%window,              -1.0)
  call readParFile(parfile, 'OFFSET_MAX',       par%offset_max,real(par%nx-1)*par%dx)
  call readParFile(parfile, 'XMAX',             par%xmax,      real(par%nx-1)*par%dx)
  
endif

!! Broadcast parameters required for all simulations
!! -------------------------------------------------
call MPI_BCAST(par%velfile,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%coordfile,100,        MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%csg_in,100,           MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%illumfile,100,        MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%qualfile,100,         MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%snapshot_file,100,    MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%nx,1,                   MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%nz,1,                   MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%ny,1,                   MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%nt,1,                   MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%dx,1,                      MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%dt,1,                      MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%fileformat,100,       MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%npml,1,                 MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%f,1,                       MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%fmax,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%strategy,100,         MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%skipshot,1,             MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%skiptrace,1,            MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%free_surface,1,         MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%sourcefile,100,       MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%sourcetype,100,       MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)

!! Broadcast parameters required for LSRTM
!! ---------------------------------------
call MPI_BCAST(par%highpass,1,             MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%migfile,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%migfile_shot,100,     MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%reflfile,100,         MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%ic,1,                   MPI_INTEGER,0,MPI_COMM_WORLD,ierr)

!! Broadcast parameters required for both LSRTM and FWI
!! ----------------------------------------------------
call MPI_BCAST(par%itermax,1,              MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%mute_direct,1,          MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%pre,1,                  MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%gradfile,100,         MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%logfile,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%n_taper,1,              MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%csg_out,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%ismarine,1,             MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%waterbottomfile,100,  MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%residualfile,100,     MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%izwb,1,                 MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%window_size,1,          MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%csg_out_res,100,      MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)

!! Broadcast parameters required for FWI 
!! -------------------------------------
call MPI_BCAST(par%nsearch_max,1,          MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%line_search,1,          MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%normalize,1,            MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%smoothgrad,1,           MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%smoothvel,1,            MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%vconstraint,1,             MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%velfile_out,100,      MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%vmin,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%vmax,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%step,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%smooth,1,               MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%perc_grad,1,               MPI_REAL,0,MPI_COMM_WORLD,ierr)

!! Broadcast parameters required for WQ 
!! ------------------------------------
call MPI_BCAST(par%csg_mod,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%tau_min,1,                 MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%tau_max,1,                 MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%smooth_tau,1,           MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%input_tau,1,            MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%taufile,100,          MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%tau_constraint,1,          MPI_REAL,0,MPI_COMM_WORLD,ierr)

!! Broadcast parameters required for multisource LSRTM 
!! ---------------------------------------------------
call MPI_BCAST(par%encoding_codes,100,   MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%encoding_polarity,100,MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)

!! Broadcast optional parameters
!! -----------------------------
call MPI_BCAST(par%velfile_bg,100,       MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%topofile,100,         MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%data,100,             MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%densityfile,100,      MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%vpfile,100,           MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%vsfile,100,           MPI_CHARACTER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%nt_in,1,                MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%nt_out,1,               MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%shift_wavefield,1,      MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%variable_density,1,     MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%mute_data,1,            MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%method,1,               MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%exportwavefield,1,      MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%dt_in,1,                   MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%dt_out,1,                  MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%xmin,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%xmax,1,                    MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%offset_min,1,              MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%offset_max,1,              MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(par%window,1,                  MPI_REAL,0,MPI_COMM_WORLD,ierr)

end subroutine readparamfile

!-------------------------------------------------------------------------------
subroutine readcoordfile(coordfile, coord)

use datatype
use mmi_mpi

character(len=*), intent(in)   :: coordfile
type(acquisition), intent(out) :: coord
integer                        :: i, j, is0, is, ig, ns, ng, ngmax
real                           :: xs, zs, xg, zg, t

if (rank == 0) then
  open(10,file=coordfile,form='formatted')

  ! Determine the number of sources
  ns = 0
  read(10,*,end=100) is0, ig, xs, zs, xg, zg, t
  ns = 1
  do i=2,1000000000
    read(10,*,end=100) is, ig, xs, zs, xg, zg, t
    if (is.ne.is0) then
      is0 = is
      ns = ns + 1
    endif
  enddo
  100 continue
  rewind(10)
  coord%ns = ns
  allocate(coord%ng(ns))
  allocate(coord%xs(ns))
  allocate(coord%zs(ns))

  ! Determine the number of geophones for each shot
  read(10,*,end=200) is0, ig, xs, zs, xg, zg, t
  ns = 1
  ng = 1
  do i=2,1000000000
    read(10,*,end=200) is, ig, xs, zs, xg, zg, t
    if (is.ne.is0) then
      is0 = is
      coord%ng(ns) = ng
!    write(*,*) 'ng(',ns,') = ', coord%ng(ns)
      ns = ns + 1
      ng = 1
    else
      ng = ng + 1
    endif
  enddo
  200 continue
  coord%ng(ns) = ng
  !write(*,*) 'ng(',ns,') = ', coord%ng(ns)
  rewind(10)

  ! Determine the maximum number of geophones per shot
  coord%ngmax = 0
  do is=1,coord%ns
    if (coord%ngmax < coord%ng(is)) coord%ngmax = coord%ng(is)
  enddo
  write(*,*) 'ng max = ', coord%ngmax

  allocate(coord%xg(is,coord%ngmax))
  allocate(coord%zg(is,coord%ngmax))
  allocate(coord%t(is,coord%ngmax))

  ! Read source and receiver positions
  do i=1,coord%ns
    do j=1,coord%ng(i)
      read(10,*,end=300) is, ig, xs, zs, xg, zg, t
      coord%xs(i) = xs
      coord%zs(i) = zs
      coord%xg(i,j) = xg
      coord%zg(i,j) = zg
      coord%t(i,j) = t
    enddo
  enddo
  300 continue
  close(10)

!  open(10,file='coord.txt',form='formatted')
!  do is=1,coord%ns
!    do ig=1,coord%ng(is)
!      write(10,*) is,ig,coord%xs(is),coord%zs(is),coord%xg(is,ig),coord%zg(is,ig),coord%t(is,ig)
!    enddo
!  enddo
!  close(10)
  write(*,*) 'ns = ', ns
endif

call MPI_BCAST(coord%ns,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%ng(coord%ns))
  allocate(coord%xs(coord%ns))
  allocate(coord%zs(coord%ns))
endif
call MPI_BCAST(coord%ng,coord%ns,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%xs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%zs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%ngmax,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%xg(coord%ns,coord%ngmax))
  allocate(coord%zg(coord%ns,coord%ngmax))
  allocate(coord%t(coord%ns,coord%ngmax))
endif
do is=1,coord%ns
  call MPI_BCAST(coord%xg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%zg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%t(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

999 continue

end subroutine readcoordfile

!-------------------------------------------------------------------------------
subroutine readcoordfile_3d(coordfile, coord)

use datatype
use mmi_mpi

character(len=*), intent(in)   :: coordfile
type(acquisition3d), intent(out) :: coord
integer                        :: i, j, is0, is, ig, ns, ng, ngmax
real                           :: xs,ys,zs,xg,yg,zg,t

if (rank == 0) then
  open(10,file=coordfile,form='formatted')

  ! Determine the number of sources
  ns = 0
  read(10,*,end=100) is0, ig, xs, ys, zs, xg, yg, zg, t
  ns = 1
  do i=2,1000000000
    read(10,*,end=100) is, ig, xs, ys, zs, xg, yg, zg, t
    if (is.ne.is0) then
      is0 = is
      ns = ns + 1
    endif
  enddo
  100 continue
  rewind(10)
  coord%ns = ns
  allocate(coord%ng(ns))
  allocate(coord%xs(ns))
  allocate(coord%ys(ns))
  allocate(coord%zs(ns))

  ! Determine the number of geophones for each shot
  read(10,*,end=200) is0, ig, xs, ys, zs, xg, yg, zg, t
  ns = 1
  ng = 1
  do i=2,1000000000
    read(10,*,end=200) is, ig, xs, ys, zs, xg, yg, zg, t
    if (is.ne.is0) then
      is0 = is
      coord%ng(ns) = ng
!    write(*,*) 'ng(',ns,') = ', coord%ng(ns)
      ns = ns + 1
      ng = 1
    else
      ng = ng + 1
    endif
  enddo
  200 continue
  coord%ng(ns) = ng
  !write(*,*) 'ng(',ns,') = ', coord%ng(ns)
  rewind(10)

  ! Determine the maximum number of geophones per shot
  coord%ngmax = 0
  do is=1,coord%ns
    if (coord%ngmax < coord%ng(is)) coord%ngmax = coord%ng(is)
  enddo
  write(*,*) 'ng max = ', coord%ngmax

  allocate(coord%xg(is,coord%ngmax))
  allocate(coord%yg(is,coord%ngmax))
  allocate(coord%zg(is,coord%ngmax))
  allocate(coord%t(is,coord%ngmax))

  ! Read source and receiver positions
  do i=1,coord%ns
    do j=1,coord%ng(i)
      read(10,*,end=300) is, ig, xs, ys, zs, xg, yg, zg, t
      coord%xs(i) = xs
      coord%ys(i) = ys
      coord%zs(i) = zs
      coord%xg(i,j) = xg
      coord%yg(i,j) = yg
      coord%zg(i,j) = zg
      coord%t(i,j) = t
    enddo
  enddo
  300 continue
  close(10)

  write(*,*) 'ns = ', ns
endif

call MPI_BCAST(coord%ns,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%ng(coord%ns))
  allocate(coord%xs(coord%ns))
  allocate(coord%ys(coord%ns))
  allocate(coord%zs(coord%ns))
endif
call MPI_BCAST(coord%ng,coord%ns,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%xs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%ys,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%zs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%ngmax,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%xg(coord%ns,coord%ngmax))
  allocate(coord%yg(coord%ns,coord%ngmax))
  allocate(coord%zg(coord%ns,coord%ngmax))
  allocate(coord%t(coord%ns,coord%ngmax))
endif
do is=1,coord%ns
  call MPI_BCAST(coord%xg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%yg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%zg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%t(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

999 continue

end subroutine readcoordfile_3d


!------------------------------------------------
subroutine read_coordfile3d_mpi(coordfile, coord)

use datatype
use mmi_mpi

character(len=*), intent(in)     :: coordfile
type(acquisition3d), intent(out) :: coord
integer                          :: is, ig
real                             :: xs, zs, xg, zg

if (rank == 0) then
  open(10,file=coordfile,form='formatted')

  !! Read ns and ngmax
  !! -----------------
  read(10,*) coord%ns, coord%ngmax
  write(*,*) 'ns = ', coord%ns
  write(*,*) 'ng max = ', coord%ngmax
  call flush(6)

  !! Allocate memory
  !! ---------------
  allocate(coord%ng(coord%ns))
  allocate(coord%xs(coord%ns))
  allocate(coord%ys(coord%ns))
  allocate(coord%zs(coord%ns))
  allocate(coord%xg(coord%ns,coord%ngmax))
  allocate(coord%yg(coord%ns,coord%ngmax))
  allocate(coord%zg(coord%ns,coord%ngmax))

  do is=1,coord%ns
    !! Read a source location and # of receivers for this source
    !! ---------------------------------------------------------
    read(10,*) coord%xs(is), coord%ys(is), coord%zs(is), coord%ng(is)
    do ig=1,coord%ng(is)
      read(10,*) coord%xg(is,ig), coord%yg(is,ig), coord%zg(is,ig)
    enddo
  enddo
endif

call MPI_BCAST(coord%ns,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%ng(coord%ns))
  allocate(coord%xs(coord%ns))
  allocate(coord%ys(coord%ns))
  allocate(coord%zs(coord%ns))
endif
call MPI_BCAST(coord%ng,coord%ns,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%xs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%ys,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%zs,coord%ns,MPI_REAL,0,MPI_COMM_WORLD,ierr)
call MPI_BCAST(coord%ngmax,1,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
if (rank > 0) then
  allocate(coord%xg(coord%ns,coord%ngmax))
  allocate(coord%yg(coord%ns,coord%ngmax))
  allocate(coord%zg(coord%ns,coord%ngmax))
endif
do is=1,coord%ns
  call MPI_BCAST(coord%xg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%yg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
  call MPI_BCAST(coord%zg(is,:),coord%ng(is),MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine read_coordfile3d_mpi

!------------------------------------------------------------------------------------
subroutine readvelfile(par,c,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
real, intent(out) :: c(:,:)
integer, intent(in) :: npml, nx_pml, nz_pml
integer :: ix, iz

if (rank == 0) then
  ! Read velocity model
  open(10,file=par%velfile,access='direct',recl=i4*par%nz)
  do ix=1,par%nx
    read(10,rec=ix) c(npml+1:npml+par%nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do ix=1,npml
    c(npml+1:npml+par%nz,ix) = c(npml+1:npml+par%nz,npml+1)
    c(npml+1:npml+par%nz,nx_pml-npml+ix) = c(npml+1:npml+par%nz,nx_pml-npml)
  enddo
  do iz=1,npml
    c(iz,:) = c(npml+1,:)
    c(nz_pml-npml+iz,:) = c(nz_pml-npml,:)
  enddo
endif
do ix=1,nx_pml
  call MPI_BCAST(c(:,ix),nz_pml,MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine readvelfile
!------------------------------------------------------------------------------------

subroutine readreflfile(par,mig,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
real, intent(out) :: mig(:,:)
integer, intent(in) :: npml, nx_pml, nz_pml
integer :: ix, iz

if (rank == 0) then
  ! Read velocity model
  open(10,file=par%reflfile,access='direct',recl=i4*par%nz,convert='little_endian')
  do ix=1,par%nx
    read(10,rec=ix) mig(npml+1:npml+par%nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do ix=1,npml
    mig(npml+1:npml+par%nz,ix) = mig(npml+1:npml+par%nz,npml+1)
    mig(npml+1:npml+par%nz,nx_pml-npml+ix) = mig(npml+1:npml+par%nz,nx_pml-npml)
  enddo
  do iz=1,npml
    mig(iz,:) = mig(npml+1,:)
    mig(nz_pml-npml+iz,:) = mig(nz_pml-npml,:)
  enddo
endif
do ix=1,nx_pml
  call MPI_BCAST(mig(:,ix),nz_pml,MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine readreflfile

!-------------------------------------------------
subroutine readqualfile(par,qf,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
real, intent(out) :: qf(:,:)
integer, intent(in) :: npml, nx_pml, nz_pml
integer :: ix, iz

if (rank == 0) then
  ! Read Q model
  open(10,file=par%qualfile,access='direct',recl=i4*par%nz,convert='little_endian')
  do ix=1,par%nx
    read(10,rec=ix) qf(npml+1:npml+par%nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate Q in PML regions
  do ix=1,npml
    qf(npml+1:npml+par%nz,ix) = qf(npml+1:npml+par%nz,npml+1)
    qf(npml+1:npml+par%nz,nx_pml-npml+ix) = qf(npml+1:npml+par%nz,nx_pml-npml)
  enddo
  do iz=1,npml
    qf(iz,:) = qf(npml+1,:)
    qf(nz_pml-npml+iz,:) = qf(nz_pml-npml,:)
  enddo
endif
do ix=1,nx_pml
  call MPI_BCAST(qf(:,ix),nz_pml,MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine readqualfile

!-------------------------------------------------
subroutine readtaufile(par,tau,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
real, intent(out) :: tau(:,:)
integer, intent(in) :: npml, nx_pml, nz_pml
integer :: ix, iz

if (rank == 0) then
  ! Read tau model
  open(10,file=par%taufile,access='direct',recl=i4*par%nz,convert='little_endian')
  do ix=1,par%nx
    read(10,rec=ix) tau(npml+1:npml+par%nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate tau in PML regions
  do ix=1,npml
    tau(npml+1:npml+par%nz,ix) = tau(npml+1:npml+par%nz,npml+1)
    tau(npml+1:npml+par%nz,nx_pml-npml+ix) = tau(npml+1:npml+par%nz,nx_pml-npml)
  enddo
  do iz=1,npml
    tau(iz,:) = tau(npml+1,:)
    tau(nz_pml-npml+iz,:) = tau(nz_pml-npml,:)
  enddo
endif
do ix=1,nx_pml
  call MPI_BCAST(tau(:,ix),nz_pml,MPI_REAL,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine readtaufile

!------------------------------------------------------------------------------------
subroutine readvelfile_new(velfile,v,nx,nz,npml)

use datatype
use global
use mmi_mpi

character(*), intent(in)  :: velfile
real,         intent(out) :: v(:,:)
integer,      intent(in)  :: nx, nz, npml
integer                   :: ix, iz, nx_pml, nz_pml

if (rank == 0) then
  nx_pml = nx+2*npml
  nz_pml = nz+2*npml

  ! Read velocity model
  open(10,file=velfile,access='direct',recl=i4*nz)
  do ix=1,nx
    read(10,rec=ix) v(npml+1:npml+nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do ix=1,npml
    v(npml+1:npml+nz,ix) = v(npml+1:npml+nz,npml+1)
    v(npml+1:npml+nz,nx_pml-npml+ix) = v(npml+1:npml+nz,nx_pml-npml)
  enddo
  do iz=1,npml
    v(iz,:) = v(npml+1,:)
    v(nz_pml-npml+iz,:) = v(nz_pml-npml,:)
  enddo
endif
!do ix=1,nx_pml
!  call MPI_BCAST(vp(:,ix),nz_pml,MPI_REAL,0,MPI_COMM_WORLD,ierr)
!enddo

end subroutine readvelfile_new

!-----------------------------------------------------------------------------------
! Read 3D velocity file in the binary format and pad the model with PML regions
! n1, n2, n3 are model dimensions without the PML region
! n1 = nz
! n2 = ny
! n3 = nx
!
subroutine read_velfile_3d(filename, v, n1, n2, n3, npml)

character(len=*), intent(in)  :: filename
integer,          intent(in)  :: n1, n2, n3, npml
real,             intent(out) :: v(:,:,:)
integer,          parameter   :: i4=4
integer                       :: i1, i2, i3

open(10,file=filename,access='direct',recl=n1*i4,status='old')

do i3=1,n3
  do i2=1,n2
    read(10,rec=i2+(i3-1)*n2) v(npml+1:npml+n1,i2+npml,i3+npml)
  enddo
enddo
close(10)

! Extrapolate velocity in PML regions
do i3=1,npml
  v(:,:,i3) = v(:,:,npml+1)
  v(:,:,n3+npml+i3) = v(:,:,n3+npml)
enddo
do i2=1,npml
  v(:,i2,:) = v(:,npml+1,:)
  v(:,n2+npml+i2,:) = v(:,n2+npml,:)
enddo
do i1=1,npml
  v(i1,:,:) = v(npml+1,:,:)
  v(n1+npml+i1,:,:) = v(n1+npml,:,:)
enddo

end subroutine read_velfile_3d

!-----------------------------------------------------------------------------------------
! Read 3D velocity file in the binary format and pad the model with PML regions
! n1, n2, n3 are model dimensions without the PML region
! n1 = nz
! n2 = ny
! n3 = nx
!
subroutine read_velfile3d_mpi(par,n1,n2,n3,npml,v)

use datatype
use mmi_mpi
use math
use global

type(param),intent(in)        :: par
integer,          intent(in)  :: n1, n2, n3, npml
real,             intent(out) :: v(:,:,:)
integer                       :: i1,i2,i3

if (rank == 0) then
  !open(10,file=filename,access='direct',recl=n1*i4,status='old')
  open(10,file=par%velfile,access='direct',recl=i4*par%nz)
  !do i3=1,n3
  !  do i2=1,n2
  do i3=1,par%nx
  do i2=1,par%ny
    !read(10,rec=i2+(i3-1)*n2) v(npml+1:npml+par%nz,i2+npml,i3+npml)
    read(10,rec=i2+(i3-1)*par%ny) v(npml+1:npml+par%nz,i2+npml,i3+npml)
  enddo
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do i3=1,npml
    v(:,:,i3) = v(:,:,npml+1)
    v(:,:,n3+npml+i3) = v(:,:,n3+npml)
  enddo
  do i2=1,npml
    v(:,i2,:) = v(:,npml+1,:)
    v(:,n2+npml+i2,:) = v(:,n2+npml,:)
  enddo
  do i1=1,npml
    v(i1,:,:) = v(npml+1,:,:)
    v(n1+npml+i1,:,:) = v(n1+npml,:,:)
  enddo
endif
call MPI_BCAST(v,(n1+2*npml)*(n2+2*npml)*(n3+2*npml),MPI_REAL,0,MPI_COMM_WORLD,ierr)
!do i3=1,n3+2*npml
!  do i2=1,n2+2*npml
!    call MPI_BCAST(v(:,i2,i3),(n1+2*npml),MPI_REAL,0,MPI_COMM_WORLD,ierr)
!  enddo
!enddo

end subroutine read_velfile3d_mpi

!-----------------------------------------------------------------------------------------
! Read 3D velocity file in the binary format and pad the model with PML regions
! n1, n2, n3 are model dimensions without the PML region
! n1 = nz
! n2 = ny
! n3 = nx
!
subroutine read_qualfile3d_mpi(par,n1,n2,n3,npml,v)

use datatype
use mmi_mpi
use math
use global

type(param),intent(in)        :: par
integer,          intent(in)  :: n1, n2, n3, npml
real,             intent(out) :: v(:,:,:)
integer                       :: i1,i2,i3

if (rank == 0) then
  open(10,file=par%qualfile,access='direct',recl=i4*par%nz)
  do i3=1,par%nx
  do i2=1,par%ny
    read(10,rec=i2+(i3-1)*par%ny) v(npml+1:npml+par%nz,i2+npml,i3+npml)
  enddo
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do i3=1,npml
    v(:,:,i3) = v(:,:,npml+1)
    v(:,:,n3+npml+i3) = v(:,:,n3+npml)
  enddo
  do i2=1,npml
    v(:,i2,:) = v(:,npml+1,:)
    v(:,n2+npml+i2,:) = v(:,n2+npml,:)
  enddo
  do i1=1,npml
    v(i1,:,:) = v(npml+1,:,:)
    v(n1+npml+i1,:,:) = v(n1+npml,:,:)
  enddo
endif
call MPI_BCAST(v,(n1+2*npml)*(n2+2*npml)*(n3+2*npml),MPI_REAL,0,MPI_COMM_WORLD,ierr)

end subroutine read_qualfile3d_mpi

!-----------------------------------------------------------
subroutine read_reflfile3d_mpi(par,n1,n2,n3,npml,v)

use datatype
use mmi_mpi
use math
use global

type(param),intent(in)        :: par
integer,          intent(in)  :: n1, n2, n3, npml
real,             intent(out) :: v(:,:,:)
integer                       :: i1,i2,i3

if (rank == 0) then
  open(10,file=par%reflfile,access='direct',recl=i4*par%nz)

  do i3=1,par%nx
  do i2=1,par%ny
    read(10,rec=i2+(i3-1)*par%ny) v(npml+1:npml+par%nz,i2+npml,i3+npml)
  enddo
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do i3=1,npml
    v(:,:,i3) = v(:,:,npml+1)
    v(:,:,n3+npml+i3) = v(:,:,n3+npml)
  enddo
  do i2=1,npml
    v(:,i2,:) = v(:,npml+1,:)
    v(:,n2+npml+i2,:) = v(:,n2+npml,:)
  enddo
  do i1=1,npml
    v(i1,:,:) = v(npml+1,:,:)
    v(n1+npml+i1,:,:) = v(n1+npml,:,:)
  enddo
endif
call MPI_BCAST(v,(n1+2*npml)*(n2+2*npml)*(n3+2*npml),MPI_REAL,0,MPI_COMM_WORLD,ierr)

end subroutine read_reflfile3d_mpi


!------------------------------------------------------------------------------------
subroutine read_densityfile(par,c,den,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi
use modeling

type(param),intent(in) :: par
real, intent(in)       :: c(:,:)
real, intent(out)      :: den(:,:)
integer, intent(in)    :: npml, nx_pml, nz_pml
integer                :: ix, iz

if (par%variable_density == 1) then
  if (par%densityfile(1:3) == 'n/a') then
    call compute_density(par%variable_density, c, den, nx_pml, nz_pml)
  else
    if (rank == 0) then
      ! Read density model
      call read_binfile_2d(par%densityfile,den(npml+1:npml+par%nz,npml+1:npml+par%nx),par%nz,par%nx)

      ! Extrapolate density in PML regions
      do ix=1,npml
        den(npml+1:npml+par%nz,ix) = den(npml+1:npml+par%nz,npml+1)
        den(npml+1:npml+par%nz,nx_pml-npml+ix) = den(npml+1:npml+par%nz,nx_pml-npml)
      enddo
      do iz=1,npml
        den(iz,:) = den(npml+1,:)
        den(nz_pml-npml+iz,:) = den(nz_pml-npml,:)
      enddo
    endif
    do ix=1,nx_pml
      call MPI_BCAST(den(:,ix),nz_pml,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
    enddo
  endif
else
  den = 1.0
endif

end subroutine read_densityfile

!------------------------------------------------------------------------------------
subroutine readvelfile_bg(par,c_bg,npml,nx_pml,nz_pml)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
real, intent(out) :: c_bg(:,:)
integer, intent(in) :: npml, nx_pml, nz_pml
integer :: ix, iz

if (rank == 0) then
  ! Read velocity model
  open(10,file=par%velfile_bg,access='direct',recl=i4*par%nz)
  do ix=1,par%nx
    read(10,rec=ix) c_bg(npml+1:npml+par%nz,ix+npml)
  enddo
  close(10)

  ! Extrapolate velocity in PML regions
  do ix=1,npml
    c_bg(npml+1:npml+par%nz,ix) = c_bg(npml+1:npml+par%nz,npml+1)
    c_bg(npml+1:npml+par%nz,nx_pml-npml+ix) = c_bg(npml+1:npml+par%nz,nx_pml-npml)
  enddo
  do iz=1,npml
    c_bg(iz,:) = c_bg(npml+1,:)
    c_bg(nz_pml-npml+iz,:) = c_bg(nz_pml-npml,:)
  enddo
endif
do ix=1,nx_pml
  call MPI_BCAST(c_bg(:,ix),nz_pml,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine readvelfile_bg

!------------------------------------------------------------------------------------
subroutine readwaterbottomfile(par,izwb)

use datatype
use global
use mmi_mpi

type(param),intent(in) :: par
integer, intent(out)   :: izwb(:)
integer                :: ix
real                   :: xwb, zwb(par%nx)

if (par%ismarine == 1) then
  if (rank == 0) then
    ! Read waterbottom file in the BINARY format
    open(10,file=par%waterbottomfile,access='direct',recl=i4*par%nx)
    read(10,rec=1) zwb
    izwb = int(zwb/par%dx)
    close(10)

!    ! Read waterbottom file in the ASCII format
!    open(10,file=par%waterbottomfile,form='formatted')
!    do ix=1,par%nx
!      read(10,*) xwb, zwb
!      izwb(ix) = int(zwb/par%dx)
!    enddo
!    close(10)
  endif
  call MPI_BCAST(izwb,par%nx,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
endif

end subroutine readwaterbottomfile

!-------------------------------------------------------------------------------
subroutine get_nt_ntrace(sufile, nt, ntrace)

use su
use global

character(len=*), intent(in) :: sufile
integer, intent(out) :: nt, ntrace
integer :: i, nlen
real, allocatable :: trace(:)

open(10,file=sufile,access='direct',recl=i4*nhead)
read(10,rec=1,err=111) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass
111 continue
close(10)
nt = ns
allocate(trace(ns))

nlen = nhead+nt
ntrace = 0
open(10,file=sufile,access='direct',recl=i4*nlen)
do i=1,10000000
  read(10,rec=i,err=222) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass,trace
  ntrace = ntrace + 1
enddo
222 continue
close(10)
write(*,*) nt, ntrace

deallocate(trace)

end subroutine get_nt_ntrace

!------------------------------------------------------------------------------
subroutine read_sufile1(sufile, trace, nt, ntrace, sampling_interval)

use global
use su

character(len=*), intent(in) :: sufile
integer, intent(in) :: nt, ntrace
real, intent(out) :: sampling_interval
real, intent(out) :: trace(:,:)
integer :: nlen, i

nlen = nhead+nt
open(10,file=sufile,access='direct',recl=i4*nlen)
do i=1,ntrace
  read(10,rec=i,err=111) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass,trace(i,1:nt)
enddo
111 continue
close(10)

sampling_interval = real(dt)/1.0e6

end subroutine read_sufile1

!------------------------------------------------------------------------------
subroutine read_sufile_2d(sufile, trace, n1, n2, dd1, dd2)

use global
use su

character(len=*), intent(in) :: sufile
integer, intent(in) :: n1, n2
real, intent(out) :: dd1, dd2
real, intent(out) :: trace(:,:)
integer :: nlen, i

nlen = nhead+n1
open(10,file=sufile,access='direct',recl=i4*nlen)
do i=1,n2
  read(10,rec=i,err=111) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass,trace(1:n1,i)
enddo
111 continue
close(10)

dd1 = real(dt)/1.0e6
dd2 = d2
if (abs(dd2) < 1.0e-6) dd2 = 1

end subroutine read_sufile_2d

!-----------------------------------------------------------------------------------
subroutine write_sufile_1d(sufile, trace, n1, dd1)

use global
use su

character(len=*), intent(in) :: sufile
integer, intent(in) :: n1
real, intent(in) :: dd1, trace(:)
integer :: nlen

call clean_suheader

nlen = nhead+n1
ns = n1
ep = 1
fldr = 1
dt = int(dd1*1.0e6)
d1 = dd1
f1 = 0
tracl = 0
tracr = 0

open(10,file=sufile,access='direct',recl=nlen*i4,status='replace')
tracl = tracl + 1
tracr = tracr + 1
cdp = 1
cdpt = 1
write(10,rec=1) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass, trace(1:n1)
close(10)

end subroutine write_sufile_1d

!-----------------------------------------------------------------------------------
subroutine write_sufile_2d(sufile, trace, n1, n2, dd1, dd2)

use global
use su

character(len=*), intent(in) :: sufile
integer, intent(in) :: n1, n2
real, intent(in) :: dd1, dd2, trace(:,:)
integer :: nlen, i

call clean_suheader

nlen = nhead+n1
ns = n1
ep = 1
fldr = 1
dt = int(dd1*1.0e6)
d1 = dd1
d2 = dd2
f1 = 0
f2 = 1.0
tracl = 0
tracr = 0

open(10,file=sufile,access='direct',recl=nlen*i4,status='replace')
do i = 1,n2
  tracl = tracl + 1
  tracr = tracr + 1
  cdp = i
  cdpt = i
  write(10,rec=i) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass, trace(1:n1,i)
enddo
close(10)

end subroutine write_sufile_2d

!-----------------------------------------------------------------------------------
subroutine write_sufile_2d_offset(sufile, starting_record, is, trace, n1, n2, dd1, dd2, coord)

use global
use datatype
use su

type(acquisition), intent(in) :: coord
character(len=*), intent(in) :: sufile
integer, intent(in) :: starting_record, is, n1, n2
real, intent(in) :: dd1, dd2, trace(:,:)
integer :: nlen, i

call clean_suheader

nlen = nhead+n1
ns = n1
ep = 1
fldr = is
dt = int(dd1*1.0e6)
d1 = dd1
d2 = dd2
f1 = 0
f2 = 1.0
tracl = 0
tracr = 0
scalel = -10
scalco = -10

open(10,file=sufile,access='direct',recl=nlen*i4)
do i = 1,n2
  tracl = tracl + 1
  tracr = tracr + 1
  cdp = i
  cdpt = i
  sx    = nint(coord%xs(is)*10.0)
  sy    = 0
  selev = nint(coord%zs(is)*10.0)
  gx    = nint(coord%xg(is,i)*10.0)
  gy    = 0
  gelev = nint(coord%zg(is,i)*10.0)
  
  write(10,rec=starting_record+i) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass, trace(1:n1,i)
enddo
close(10)

end subroutine write_sufile_2d_offset

!-----------------------------------------------------------------------------------
subroutine write_sufile_2d_coord(sufile, trace, n1, n2, dd1, dd2, xs, xg)

use global
use su

character(len=*), intent(in) :: sufile
integer, intent(in) :: n1, n2
real, intent(in) :: dd1, dd2, trace(:,:), xs, xg(n2)
integer :: nlen, i

call clean_suheader

nlen = nhead+n1
ns = n1
ep = 1
fldr = 1
dt = int(dd1*1.0e6)
d1 = dd1
d2 = dd2
f1 = 0
f2 = 1.0
tracl = 0
tracr = 0
sx = nint(xs)

open(10,file=sufile,access='direct',recl=nlen*i4,status='replace')
do i = 1,n2
  tracl = tracl + 1
  tracr = tracr + 1
!  cdp = i
!  cdpt = i
  gx = nint(xg(i))
  cdp = nint(0.5*(xs+xg(i)))
  cdpt = cdp
  offset = gx - sx
  write(10,rec=i) &
        tracl,tracr,fldr,tracf,ep,cdp,cdpt, &
        trid,nvs,nhs,duse,offset,gelev,selev,sdepth, &
        gdel,sdel,swdep,gwdep,scalel,scalco, &
        sx,sy,gx,gy,counit,wevel,swevel,sut,gut,sstat,gstat, &
        tstat,laga,lagb,delrt,muts,mute,ns,dt, &
        gain,igc,igi,corr,sfs,sfe,slen,styp,stas, &
        stae,tatyp,afilf,afils,nofilf,nofils,lcf,hcf, &
        lcs,hcs,year,day,hour,minute,sec,timbas, &
        trwf,grnors,grnofr,grnlof,gaps,otrav, &
        d1,f1,d2,f2,ungpow,unscale,mark,unass, trace(1:n1,i)
enddo
close(10)

end subroutine write_sufile_2d_coord
!-----------------------------------------------------------------------------------

subroutine write_binfile_1d(filename, data, n)

use global

character(len=*), intent(in) :: filename
integer, intent(in) :: n
real, intent(in) :: data(:)
integer :: i,it

open(10,file=filename,access='direct',recl=n*i4,status='replace')
write(10,rec=1) data(1:n)
close(10)

end subroutine write_binfile_1d
!-----------------------------------------------------------------------------------

subroutine write_binfile_2d(filename, data, n1, n2)

use global

character(len=*), intent(in) :: filename
integer, intent(in) :: n1, n2
real, intent(in) :: data(:,:)
integer :: i,it

open(10,file=filename,access='direct',recl=n1*i4,status='replace')

do i = 1, n2
   write(10,rec=i)(data(it,i), it = 1, n1)
enddo
close(10)

end subroutine write_binfile_2d

!-----------------------------------------------------------------------------------
subroutine write_binfile_3d(filename, data, n1, n2, n3)

character(len=*), intent(in) :: filename
integer,          intent(in) :: n1, n2, n3
real,             intent(in) :: data(:,:,:)
integer,          parameter  :: i4=4
integer                      :: i1, i2, i3

open(10,file=filename,access='direct',recl=n1*i4,status='replace')

do i3=1,n3
  do i2=1,n2
    write(10,rec=i2+(i3-1)*n2) data(:,i2,i3)
  enddo
enddo
close(10)

end subroutine write_binfile_3d

!-----------------------------------------------------------------------------------
subroutine write_binfile_intel_1d(filename, data, n)

use global

character(len=*), intent(in) :: filename
integer, intent(in) :: n
real, intent(in) :: data(:)
integer :: i,it

open(10,file=filename,access='direct',recl=n*i4,status='replace')
write(10,rec=1) data(1:n)
close(10)

end subroutine write_binfile_intel_1d

!-----------------------------------------------------------------------------------

subroutine write_binfile_intel_2d(filename, data, n1, n2)

use global

character(len=*), intent(in) :: filename
integer, intent(in) :: n1, n2
real, intent(in) :: data(:,:)
integer :: i,it

open(10,file=filename,access='direct',recl=n1*n2*i4,status='replace')

write(10,rec=1) data
close(10)

end subroutine write_binfile_intel_2d

!-----------------------------------------------------------------------------------
subroutine write_binfile_intel_3d(filename, data, n1, n2, n3)

character(len=*), intent(in) :: filename
integer,          intent(in) :: n1, n2, n3
real,             intent(in) :: data(:,:,:)
integer,          parameter  :: i4=4
integer                      :: i1, i2, i3

open(10,file=filename,access='direct',recl=n1*n2*n3*i4,status='replace')

write(10,rec=1) data
close(10)

end subroutine write_binfile_intel_3d

!-----------------------------------------------------------------------------------
subroutine read_binfile_1d(filename, data, n)

use global

implicit none

character(len=*), intent(in) :: filename
integer, intent(in)          :: n
real                         :: data(:)

open(10,file=filename,access='direct',recl=n*i4)
read(10, rec=1) data(1:n)
close(10)

end subroutine read_binfile_1d

!-----------------------------------------------------------------------------------
subroutine read_binfile_2d(filename, data, n1, n2)

use global

implicit none

character(len=*), intent(in) :: filename
integer, intent(in) :: n1, n2
real                :: data(:,:)
integer :: i1,i

open(10,file=filename,access='direct',recl=n1*i4)

do i = 1, n2
   read(10, rec=i)(data(i1,i),i1=1,n1)
enddo
close(10)

end subroutine read_binfile_2d

!-----------------------------------------------------------------------------------
subroutine read_binfile_3d(filename, data, n1, n2, n3)

character(len=*), intent(in)  :: filename
integer,          intent(in)  :: n1, n2, n3
real,             intent(out) :: data(:,:,:)
integer,          parameter   :: i4=4
integer                       :: i1, i2, i3

open(10,file=filename,access='direct',recl=n1*i4,status='old')

do i3=1,n3
  do i2=1,n2
    read(10,rec=i2+(i3-1)*n2) data(:,i2,i3)
  enddo
enddo
close(10)

end subroutine read_binfile_3d

!-----------------------------------------------------------------------------------
subroutine read_binfile_intel_1d(filename, data, n)

use global

implicit none

character(len=*), intent(in) :: filename
integer, intent(in)          :: n
real                         :: data(:)

open(10,file=filename,access='direct',recl=n*i4)
read(10, rec=1) data(1:n)
close(10)

end subroutine read_binfile_intel_1d

!-----------------------------------------------------------------------------------
subroutine read_binfile_intel_2d(filename, data, n1, n2)

use global

implicit none

character(len=*), intent(in) :: filename
integer, intent(in) :: n1, n2
real                :: data(:,:)
integer :: i1,i

open(10,file=filename,access='direct',recl=n1*n2*i4)
read(10, rec=1) data
close(10)

end subroutine read_binfile_intel_2d

!-----------------------------------------------------------------------------------
subroutine read_binfile_intel_3d(filename, data, n1, n2, n3)

character(len=*), intent(in)  :: filename
integer,          intent(in)  :: n1, n2, n3
real,             intent(out) :: data(:,:,:)
integer,          parameter   :: i4=4
integer                       :: i1, i2, i3

open(10,file=filename,access='direct',recl=n1*n2*n3*i4,status='old')
read(10,rec=1) data
close(10)

end subroutine read_binfile_intel_3d



!-----------------------------------------------------------------------------------
subroutine clean_suheader

use su
implicit none

integer i

tracl = 0
tracr = 0
fldr = 0
tracf = 0
ep = 0
cdp = 0
cdpt = 0
trid = 0
nvs = 0
nhs = 0
duse = 0
offset = 0
gelev = 0
selev = 0
sdepth = 0
gdel = 0
sdel = 0
swdep = 0
gwdep = 0
scalel = 0
scalco = 0
sx = 0
sy = 0
gx = 0
gy = 0
counit = 0
wevel = 0
swevel = 0
sut = 0
gut = 0
sstat = 0
gstat = 0
tstat = 0
laga = 0
lagb = 0
delrt = 0
muts = 0
mute = 0
ns = 0
dt = 0
gain = 0
igc = 0
igi = 0
corr = 0
sfs = 0
sfe = 0
slen = 0
styp = 0
stas = 0
stae = 0
tatyp = 0
afilf = 0
afils = 0
nofilf = 0
nofils = 0
lcf = 0
hcf = 0
lcs = 0
hcs = 0
year = 0
day = 0
hour = 0
minute = 0
sec = 0
timbas = 0
trwf = 0
grnors = 0
grnofr = 0
grnlof = 0
gaps = 0
otrav = 0
d1 = 0.0
f1 = 0.0
d2 = 0.0
f2 = 0.0
ungpow = 0.0
unscale = 0.0
mark = 0
unass(:) = 0

end subroutine clean_suheader

!-----------------------------------------------------------------------------------
subroutine free_surface(par, fs, npml)

use datatype
use global
use mmi_mpi

type(param), intent(in) :: par
integer, intent(in)     :: npml
integer, intent(out)    :: fs(:)
integer                 :: nx_pml, ix
real                    :: topo(par%nx)

nx_pml = par%nx + 2*npml
if (rank == 0) then
  open(10,file=par%topofile,access='direct',recl=par%nx*i4,err=111)
  read(10,rec=1) topo
  close(10)
  goto 222
  111 continue
  topo = 0.0
  222 continue

  ! Set up free surface
  fs(npml+1:nx_pml-npml) = npml+1+int(topo/par%dx)
  fs(1:npml) = fs(npml+1)
  fs(nx_pml-npml+1:nx_pml) = fs(nx_pml-npml)
endif
call MPI_BCAST(fs,nx_pml,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)

end subroutine free_surface

!-------------------------------------
subroutine free_surface_3d(par,fs,npml)

use datatype
use global
use mmi_mpi

type(param), intent(in) :: par
integer, intent(in)     :: npml
integer, intent(out)    :: fs(:,:)
integer                 :: nx_pml,ny_pml,ix,iy
real, allocatable       :: topo(:,:)


nx_pml=par%nx+2*npml
ny_pml=par%ny+2*npml

if (rank.eq.0) then

  allocate(topo(par%ny,par%nx))
  open(10,file=par%topofile,access='direct',recl=par%ny*i4,err=111)
  do ix=1,par%nx
    read(10,rec=ix) topo(:,ix)
  enddo
  close(10)
  goto 222
  111 continue
  topo = 0.0
  222 continue

  !! Set up free surface
  !! -------------------
  fs = 0
  do ix=1,par%nx
  do iy=1,par%ny
    fs(npml+iy,npml+ix)=npml+int(topo(iy,ix)/par%dx)+1
  enddo
  enddo

  do ix=1,nx_pml
  do iy=1,npml
    fs(iy,ix)=fs(npml+1,ix)
    fs(ny_pml-iy,ix)=fs(ny_pml-npml,ix)
  enddo
  enddo

  do ix=1,npml
  do iy=1,ny_pml
    fs(iy,ix)=fs(iy,npml+1)
    fs(iy,nx_pml-ix)=fs(iy,nx_pml-npml)
  enddo
  enddo
  deallocate(topo)
endif

do iy=1,ny_pml
  call MPI_BCAST(fs(:,iy),nx_pml,MPI_INTEGER,0,MPI_COMM_WORLD,ierr)
enddo

end subroutine free_surface_3d

!-----------------------------------------------------------------------------------
subroutine writehead(headername,binaryname,esize,data_format,&
                     n1, n2, n3, n4, n5,                     &
                     d1, d2, d3, d4, d5,                     &
                     o1, o2, o3, o4, o5,                     &
                     l1, l2, l3, l4, l5)

    character(len=*),       intent(inout) :: headername
    character(len=*),optional,intent(in)  :: binaryname
    integer,         optional,intent(in)  :: esize
    integer,         optional,intent(in)  :: n1, n2, n3, n4, n5
    real,            optional,intent(in)  :: d1, d2, d3, d4, d5
    real,            optional,intent(in)  :: o1, o2, o3, o4, o5
    character(len=*),optional,intent(in)  :: l1, l2, l3, l4, l5
    character(len=*),optional,intent(in)  :: data_format

    character(len=512) :: tmp_str
    integer            :: iunit,maxdim
    character(len=32)  :: local_data_format
    integer            :: ierr

    ierr=0

    iunit=96

    maxdim=5

    open(unit=iunit,file=trim(headername),form='formatted',position='append',iostat=ierr)

    write(iunit,'(a)') '#SEP'

    if (present(n1)) then
       write(tmp_str,*) n1
       write(iunit,'(a)') 'n1='//trim(adjustl(tmp_str))
    endif

    if (present(n2).and.(maxdim.ge.2)) then
       write(tmp_str,*) n2
       write(iunit,'(a)') 'n2='//trim(adjustl(tmp_str))
    endif

    if (present(n3).and.(maxdim.ge.3)) then
       write(tmp_str,*) n3
       write(iunit,'(a)') 'n3='//trim(adjustl(tmp_str))
    endif

    if (present(n4).and.(maxdim.ge.4)) then
       write(tmp_str,*) n4
       write(iunit,'(a)') 'n4='//trim(adjustl(tmp_str))
    endif

    if (present(n5).and.(maxdim.ge.5)) then
       write(tmp_str,*) n5
       write(iunit,'(a)') 'n5='//trim(adjustl(tmp_str))
    endif

    if (present(d1)) then
       write(tmp_str,*) d1
       write(iunit,'(a)') 'd1='//trim(adjustl(tmp_str))
    endif

    if (present(d2).and.(maxdim.ge.2)) then
       write(tmp_str,*) d2
       write(iunit,'(a)') 'd2='//trim(adjustl(tmp_str))
    endif

    if (present(d3).and.(maxdim.ge.3)) then
       write(tmp_str,*) d3
       write(iunit,'(a)') 'd3='//trim(adjustl(tmp_str))
    endif

    if (present(d4).and.(maxdim.ge.4)) then
       write(tmp_str,*) d4
       write(iunit,'(a)') 'd4='//trim(adjustl(tmp_str))
    endif

    if (present(d5).and.(maxdim.ge.5)) then
       write(tmp_str,*) d5
       write(iunit,'(a)') 'd5='//trim(adjustl(tmp_str))
    endif

    if (present(o1)) then
       write(tmp_str,*) o1
       write(iunit,'(a)') 'o1='//trim(adjustl(tmp_str))
    endif

    if (present(o2).and.(maxdim.ge.2)) then
       write(tmp_str,*) o2
       write(iunit,'(a)') 'o2='//trim(adjustl(tmp_str))
    endif

    if (present(o3).and.(maxdim.ge.3)) then
       write(tmp_str,*) o3
       write(iunit,'(a)') 'o3='//trim(adjustl(tmp_str))
    endif

    if (present(o4).and.(maxdim.ge.4)) then
       write(tmp_str,*) o4
       write(iunit,'(a)') 'o4='//trim(adjustl(tmp_str))
    endif

    if (present(o5).and.(maxdim.ge.5)) then
       write(tmp_str,*) o5
       write(iunit,'(a)') 'o5='//trim(adjustl(tmp_str))
    endif

    if (present(l1)) then
       write(iunit,'(a)') 'label1="'//trim(l1)//'"'
    endif

    if (present(l2).and.(maxdim.ge.2)) then
       write(iunit,'(a)') 'label2="'//trim(l2)//'"'
    endif

    if (present(l3).and.(maxdim.ge.3)) then
       write(iunit,'(a)') 'label3="'//trim(l3)//'"'
    endif

    if (present(l4).and.(maxdim.ge.4)) then
       write(iunit,'(a)') 'label4="'//trim(l4)//'"'
    endif

    if (present(l5).and.(maxdim.ge.5)) then
       write(iunit,'(a)') 'label5="'//trim(l5)//'"'
    endif

    write(iunit,'(a)') 'esize='//trim(adjustl("4"))

    write(iunit,'(a)') 'in='//trim(adjustl(headername))//"@"

    local_data_format="native_float"
    write(iunit,'(a)') 'data_format='//trim(adjustl(local_data_format))

    flush(iunit)

    close(iunit,iostat=ierr)

    return

  end subroutine writehead


end module io

