! module for defining PML arrays
!
module pml

implicit none
integer :: npml,nx_pml,ny_pml,nz_pml,iz1,iz2,ix1,ix2,iy1,iy2
real, allocatable :: damp(:),damp_global(:,:),damp_global_3d(:,:,:)

contains

subroutine init_pml(nx,nz,n_pml)

integer, intent(in)          :: nx, nz, n_pml

npml = n_pml
nx_pml = nx+2*npml
nz_pml = nz+2*npml
ix1 = npml+1
ix2 = npml+nx
iz1 = npml+1
iz2 = npml+nz
allocate(damp(npml))
allocate(damp_global(nz_pml,nx_pml))

end subroutine init_pml

!------------------------------------
subroutine init_pml_3d(nx,ny,nz,n_pml)

integer, intent(in)          :: nx,ny,nz,n_pml

npml = n_pml
nx_pml = nx+2*npml
ny_pml = ny+2*npml
nz_pml = nz+2*npml
ix1 = npml+1
ix2 = npml+nx
iy1 = npml+1
iy2 = npml+ny
iz1 = npml+1
iz2 = npml+nz
allocate(damp(npml))
allocate(damp_global_3d(nz_pml,ny_pml,nx_pml))

end subroutine init_pml_3d


!----------------------------------------------------------------------------
subroutine setup_damp_compact(nx_compact, nz_compact, dx, cmin, damp_compact)

integer, intent(in) :: nx_compact, nz_compact
real, intent(in)    :: dx, cmin
real, intent(out)   :: damp_compact(:,:)
real                :: a, xa, kappa
integer             :: ix, iz, nx_compact_pml, nz_compact_pml

nx_compact_pml = nx_compact + 2*npml
nz_compact_pml = nz_compact + 2*npml
damp_compact = 0.0
a = (npml-1)*dx
kappa = 15.0*cmin/(2.0*a)
do ix=1,npml
  xa = real(ix-1)*dx/a
  damp(ix) = kappa*xa*xa
enddo
do ix=1,npml
  do iz=1,nz_compact_pml
    damp_compact(iz,npml-ix+1) = damp(ix)
    damp_compact(iz,nx_compact_pml+ix-1-npml) = damp(ix)
  enddo
enddo
do iz=1,npml
  do ix=1+(npml-(iz-1)),nx_compact_pml-(npml-(iz-1))
    damp_compact(npml-iz+1,ix) = damp(iz)
    damp_compact(nz_compact_pml+iz-1-npml,ix) = damp(iz)
  enddo
enddo

end subroutine setup_damp_compact

!-------------------------------------------------------------------
subroutine setup_pml(dx, cmin)

real, intent(in) :: dx, cmin
real             :: a, xa, kappa
integer          :: ix, iz

damp_global = 0.0
a = (npml-1)*dx
kappa = 3.0*cmin*log(10000000.0)/(2.0*a)
do ix=1,npml
  xa = real(ix-1)*dx/a
  damp(ix) = kappa*xa*xa
enddo
do ix=1,npml
  do iz=1,nz_pml
    damp_global(iz,npml-ix+1) = damp(ix)
    damp_global(iz,nx_pml+ix-npml) = damp(ix)
  enddo
enddo
do iz=1,npml
  do ix=1+(npml-(iz-1)),nx_pml-(npml-(iz-1))
    damp_global(npml-iz+1,ix) = damp(iz)
    damp_global(nz_pml+iz-npml,ix) = damp(iz)
  enddo
enddo

end subroutine setup_pml

!-------------------------------
subroutine setup_pml_3d(nx,ny,nz,dx,cmin)

integer, intent(in) :: nx,ny,nz
real,   intent(in)  :: dx,cmin
integer             :: ix,iy,iz
real                :: a,xa,kappa

!! Setup 1D PML damping term
!! -------------------------
a=(npml-1)*dx
kappa=3.0*cmin*log(10000000.0)/(2.0*a)
do ix=1,npml
    xa=real(ix-1)*dx/a
    damp(ix)=kappa*xa*xa
enddo

!! Setup 3D PML damping array
!! --------------------------
damp_global_3d = 0.0
iy = 1
do ix=1,npml
  damp_global_3d(:,:,npml-ix+1) = damp(ix)
  damp_global_3d(:,:,nx+ix-npml) = damp(ix)
enddo
do iz=1,npml
do ix=1+(npml-(iz-1)),nx-(npml-(iz-1))
    damp_global_3d(npml-iz+1,:,ix) = damp(iz)
    damp_global_3d(nz+iz-npml,:,ix) = damp(iz)
enddo
enddo
do iy=1,npml
do ix=1+(npml-(iy-1)),nx-(npml-(iy-1))
do iz=1+(npml-(iy-1)),nz-(npml-(iy-1))
    damp_global_3d(iz,npml-iy+1,ix) = damp(iy)
    damp_global_3d(iz,ny+iy-npml,ix) = damp(iy)
enddo
enddo
enddo

deallocate(damp)
end subroutine setup_pml_3d

!-----------------------------------------------------------------------------------------
subroutine pml_get_damp3d(nx, ny, nz, npml, dx, cmin, damp)

integer, intent(in)  :: nx, ny, nz, npml
real,    intent(in)  :: dx, cmin
real,    intent(out) :: damp(:,:,:)

integer              :: ix, iy, iz
real                 :: a, xa, kappa
real, allocatable    :: damp1d(:)

allocate(damp1d(npml))

! Setup 1D PML damping term
a = (npml-1)*dx
kappa = 3.0*cmin*log(10000000.0)/(2.0*a)
do ix=1,npml
  xa = real(ix-1)*dx/a
  damp1d(ix) = kappa*xa*xa
enddo

! Setup 3D PML damping array
damp = 0.0
iy = 1
do ix=1,npml
   damp(:,:,npml-ix+1) = damp1d(ix)
   damp(:,:,nx+ix+npml) = damp1d(ix)
enddo
do iz=1,npml
   do ix=npml-(iz-1),nx+npml+(iz-1)
      damp(npml-iz+1,:,ix) = damp1d(iz)
      damp(nz+iz+npml,:,ix) = damp1d(iz)
   enddo
enddo
!$OMP PARALLEL DO PRIVATE(iz,iy,ix)
do iy=1,npml
   do ix=npml-(iy-1),nx+npml+(iy-1)
      do iz=npml-(iy-1),nz+npml+(iy-1)
         damp(iz,npml-iy+1,ix) = damp1d(iy)
         damp(iz,ny+iy+npml,ix) = damp1d(iy)
      enddo
   enddo
enddo
!$OMP END PARALLEL DO

deallocate(damp1d)

end subroutine pml_get_damp3d

!------------------------------------------------------------------------------------
! m is the model to be padded
subroutine padmodel(par,m,npml,nx_pml,nz_pml)

use datatype
!use global

type(param),intent(in)  :: par
integer,    intent(in)  :: npml, nx_pml, nz_pml
real,       intent(out) :: m(:,:)
integer :: ix, iz

! Extrapolate velocity in PML regions
do ix=1,npml
  m(npml+1:npml+par%nz,ix) = m(npml+1:npml+par%nz,npml+1)
  m(npml+1:npml+par%nz,nx_pml-npml+ix) = m(npml+1:npml+par%nz,nx_pml-npml)
enddo
do iz=1,npml
  m(iz,:) = m(npml+1,:)
  m(nz_pml-npml+iz,:) = m(nz_pml-npml,:)
enddo

end subroutine padmodel
!------------------------------------------------------------------------------------

end module pml

